/** @file   objectstorage.cpp
 * @brief   Implementation of ObjectStorage - class.
 * @version $Revision: 1.4 $
 * @author  Tomi Lamminsaari
 */

#include "objectstorage.h"
#include "www_assert.h"
#include "eng2d.h"
using std::vector;
using std::multimap;
using std::pair;
using std::string;
using eng2d::Rect2D;

namespace WeWantWar {

/** Constructor
 */
ObjectStorage::ObjectStorage() :
  pPlayer( 0 ),
  pPrimaryTarget( 0 )
{
}



/** Destructor
 */
ObjectStorage::~ObjectStorage()
{
  this->clear();
}




///
/// Public methods
/// ==============

/** Add the given gameobject.
 */
void ObjectStorage::addObject( GameObject* pO )
{
  // We add it always to the objectlist. */
  objectList.push_back( pO );
  
  // Now we check the type of this object and add it to the right table. */
  if ( pO->state() == GameObject::STATE_KILLED ) {
    // This object is already killed.
    killedObjects.push_back( pO );
    
  } else {
    switch ( pO->objectType() ) {
      case ( ObjectID::TYPE_PLAYER ): {
        pPlayer = dynamic_cast<Player*>( pO );
        break;
      }
      case ( ObjectID::TYPE_CARNIVOREALIEN ):
      case ( ObjectID::TYPE_SMALLWORMALIEN ):
      case ( ObjectID::TYPE_WINGEDALIEN ):
      case ( ObjectID::TYPE_PROCTORALIEN ):
      case ( ObjectID::TYPE_MINIGUNALIEN ):
      case ( ObjectID::TYPE_PREDATORALIEN ):
      case ( ObjectID::TYPE_SENTRYGUN ):
      case ( ObjectID::TYPE_FIGHTER ): {
        hostileObjects.push_back( pO );
        break;
      }
      case ( ObjectID::TYPE_PLAYERCAR ):
      case ( ObjectID::TYPE_CAR ):
      case ( ObjectID::TYPE_TANK ): {
        vehicleObjects.push_back( pO );
        break;
      }
      case ( ObjectID::TYPE_CIVILIAN ): {
        civilianObjects.push_back( pO );
        break;
      }
    }
    
  }
  
  // Add the object to the multimap also.
  pair< GameObject::IDCode, GameObject* > p;
  p.first = pO->objectID();
  p.second = pO;
  objectByID.insert( p );
}



/** Removes the objects from the tables.
 */
void ObjectStorage::remObject( GameObject* pO )
{
  // Remove the object from the objecttable.
  for ( vector<GameObject*>::iterator it = objectList.begin();
        it != objectList.end(); it++ ) {
    if ( *it == pO ) {
      // We found the object
      objectList.erase( it );
      break;
    }
  }
  delete pO;
}



/** Removes the object at position
 */
void ObjectStorage::remObject( int pos )
{
  WWW_ASSERT( pos >= 0 );
  WWW_ASSERT( pos < objectList.size() );
  
  GameObject* pO = objectList.at( pos );
  this->remObject( pO );
}



/** Updates all the objects
 */
void ObjectStorage::updateObjects()
{
  for ( int i=0; i < objectList.size(); i++ ) {
    objectList.at(i)->updateObject();
  }
}



/** Redraws all the objects
 */
void ObjectStorage::redrawObjects( RedrawQueue* pQueue )
{
  for ( int i=0; i < objectList.size(); i++ ) {
    if ( objectList.at(i) != 0 ) {
    #ifdef NDEBUG
      objectList.at(i)->redraw( pQueue );
      
    #else
      try {
        objectList.at(i)->redraw( pQueue );
      } catch (...) {
        string mess = eng2d::Int2String( i );
        string mess2 = eng2d::Int2String( objectList.at(i)->objectType() );
        alert( "ObjectStorage::redrawObjects()",
               mess.c_str(), mess2.c_str(), "ok",0, 0,0 );
        abort();
      }
    #endif
    }
  }
}



/** Removes all the objects.
 */
void ObjectStorage::clear()
{
  for ( vector<GameObject*>::iterator it = objectList.begin();
        it != objectList.end(); it++ ) {
    delete *it;
    *it = 0;
  }
  objectList.clear();
  killedObjects.clear();
  civilianObjects.clear();
  vehicleObjects.clear();
  hostileObjects.clear();
  objectByID.clear();
  
  pPlayer = 0;
  
  pPrimaryTarget = 0;
  secondaryTargets.clear();
}



///
/// Getter methods
/// ==============

/** Tells if there is an object with given id
 */
bool ObjectStorage::hasObjectWithID( GameObject::IDCode id ) const
{
  pair< IDTable::const_iterator, IDTable::const_iterator >
    lims = objectByID.equal_range( id );
  if ( lims.first == objectByID.end() ) {
    return false;
  }
  return true;
}



/** Returns all the objects that have given ID-code.
 */
bool ObjectStorage::getObjectsWithID( GameObject::IDCode id,
                                      vector<GameObject*>& rV ) const
{
  rV.clear();
  
  // Search the range where the objects with given ID-codes can be found.
  std::pair< IDTable::const_iterator, IDTable::const_iterator > iters =
    objectByID.equal_range( id );
    
  // Push the found pointers to the given vector.
  for ( IDTable::const_iterator it = iters.first; it != iters.second; it++ ) {
    rV.push_back( it->second );
  }
  if ( rV.size() == 0 ) {
    return false;
  }
  return true;
}



/** Tells if there are objects inside the given rectanlge.
 */
bool ObjectStorage::objectsInsideRect( const Rect2D& rRect ) const
{
  for ( int i=0; i < objectList.size(); i++ ) {
    if ( rRect.pointInside( objectList.at(i)->position() ) == true ) {
      return true;
    }
  }
  return false;
}


GameObject* ObjectStorage::findObject( GameObject::IDCode aId ) const
{
  vector<GameObject*> vec;
  if ( this->getObjectsWithID(aId, vec) == false ) {
    return 0;
  }
  return vec.at(0);
}




///
/// Private or Protected methods
/// ============================

} // end of namespace
